﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net.Http;
using System.Text.Json;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Documents;
using System.Windows.Media;
using Path = System.IO.Path;

namespace LoadProfiles;

/// <summary>
/// Interaction logic for MainWindow.xaml
/// </summary>
public partial class MainWindow : Window
{
    private const string ProfileSettingsJsonFileName = "ProfileSettings.json";

    private string ProfileSettingsJsonPath
    {
        get
        {
            var appPath = Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData);
            var settingsPath = Path.Combine(appPath, "LoadProfiles");
            if (!Directory.Exists(settingsPath))
            {
                Directory.CreateDirectory(settingsPath);
            }
            return Path.Combine(settingsPath, ProfileSettingsJsonFileName);
        }
    }

    private readonly Settings _settings;

    public MainWindow()
    {
        InitializeComponent();
        _settings = LoadSettings();
        LoadHistory(_settings.Profiles);
    }

    private void LoadHistory(List<string> profiles)
    {
        for (var index = profiles.Count - 1; index >= 0; index--)
        {
            var userName = profiles[index];
            AddHistory(userName);
        }
    }


    private async void Lookup_Click(object sender, RoutedEventArgs e)
    {
        if (!string.IsNullOrWhiteSpace(UsernameTextBox.Text))
        {
            await DoSearchAndOpenTab(UsernameTextBox.Text);
        }
    }

    private async Task DoSearchAndOpenTab(string text)
    {
        try
        {
            var profile = await SearchHelper.DoSearch(text);
            if (profile != null)
            {
                Dispatcher.Invoke(() => AddHistoryAndTab(profile));
            }
            else
            {
                Dispatcher.Invoke(ProfileNotFound);
            }
        }
        catch (HttpRequestException ex)
        {
            Dispatcher.Invoke(() => ShowError(ex));
        }
    }

    private void ShowError(HttpRequestException ex)
    {
        MessageBox.Show(this, $"Got exception fetching from server: {ex.StatusCode?.ToString("D")}/{ex.Message}", "Error", MessageBoxButton.OK, MessageBoxImage.Error);
    }

    private void ProfileNotFound()
    {
        ErrorNotFound.Visibility = Visibility.Visible;
    }

    private void AddHistoryAndTab(Profile profile)
    {
        ErrorNotFound.Visibility = Visibility.Hidden;
        UpdateSettings(profile.UserName);
        AddHistory(profile.UserName);
        AddTab(profile);
    }

    private void AddTab(Profile profile)
    {
        var tabItem = new TabItem();
        var button = new Button { Content = "×", BorderBrush = Brushes.Transparent, Background = Brushes.Transparent };
        button.Click += (sender, args) => { TabControl.Items.Remove(tabItem); };
        var tabTitle = new TextBlock { Text = profile.UserName, VerticalAlignment = VerticalAlignment.Center };
        tabItem.Header = new StackPanel
        {
            Orientation = Orientation.Horizontal,
            Children = { tabTitle, button }
        };

        var profileControl = new ProfileControl
        {
            DataContext = profile
        };

        tabItem.Content = profileControl;

        var idx = TabControl.Items.Add(tabItem);
        TabControl.SelectedIndex = idx;
    }

    private void AddHistory(string userName)
    {
        var existingLink = FindExistingLink(userName);
        if (existingLink != null)
        {
            HistoryPanel.Children.Remove(existingLink);
            HistoryPanel.Children.Insert(0, existingLink);
        }
        else
        {
            InsertHistory(userName);
        }
    }

    private void InsertHistory(string userName)
    {
        var linkText = new Run(userName);
        var link = new Hyperlink(linkText)
        {
            DataContext = userName
        };
        link.Click += Link_Click;
        var label = new Label
        {
            Content = link
        };

        HistoryPanel.Children.Insert(0, label);
    }

    private Label? FindExistingLink(string userName)
    {
        foreach (Label label in HistoryPanel.Children)
        {
            var link = (Hyperlink)label.Content;
            if (userName == link.DataContext.ToString())
            {
                return label;
            }
        }

        return null;
    }

    private void UpdateSettings(string userName)
    {
        if (_settings.Profiles.Contains(userName))
        {
            _settings.Profiles.Remove(userName);
        }

        _settings.Profiles.Insert(0, userName);
    }

    private async void Link_Click(object sender, RoutedEventArgs e)
    {
        var link = (Hyperlink)sender;
        var text = link.DataContext.ToString();
        if (!string.IsNullOrEmpty(text))
        {
            await DoSearchAndOpenTab(text);
        }
    }

    private void Window_Closing(object sender, System.ComponentModel.CancelEventArgs e)
    {
        SaveSettings();
    }

    private Settings LoadSettings()
    {
        if (!File.Exists(ProfileSettingsJsonPath))
        {
            return new Settings();
        }
        var contents = File.ReadAllText(ProfileSettingsJsonPath);
        if (string.IsNullOrWhiteSpace(contents))
        {
            return new Settings();
        }

        return JsonSerializer.Deserialize<Settings>(contents) ?? new Settings();
    }

    private void SaveSettings()
    {
        var contents = JsonSerializer.Serialize(_settings);

        File.WriteAllText(ProfileSettingsJsonPath, contents);
    }

    private void Sort_Ascending_Click(object sender, RoutedEventArgs e)
    {
        HistoryPanel.Children.Clear();
        LoadHistory(_settings.Profiles.OrderBy(s => s).ToList());
    }

    private void Sort_Descending_Click(object sender, RoutedEventArgs e)
    {
        HistoryPanel.Children.Clear();
        LoadHistory(_settings.Profiles.OrderByDescending(s => s).ToList());
    }

    private void Clear_Sort_Click(object sender, RoutedEventArgs e)
    {
        HistoryPanel.Children.Clear();
        LoadHistory(_settings.Profiles);
    }

    private void Clear_Click(object sender, RoutedEventArgs e)
    {
        HistoryPanel.Children.Clear();
        _settings.Profiles.Clear();
    }
}

public class Settings
{
    public List<string> Profiles { get; set; } = new();
}