﻿using System;
using System.Linq;
using System.Net.Http;
using System.Text;
using System.Text.Json;
using System.Threading.Tasks;
using System.Web;
using System.Windows.Input;
using HtmlAgilityPack;

namespace LoadProfiles;

public static class SearchHelper
{
    public static async Task<Profile?> DoSearch(string userName)
    {
        using var client = new HttpClient();
        var redirectedUrl = await GetRedirectedUrl($"https://literotica.com/authors/{userName}");

        if (redirectedUrl == null)
        {
            return null;
        }

        var responseString = await client.GetStringAsync(redirectedUrl);
        var doc = new HtmlDocument();
        doc.LoadHtml(responseString);
        var docNode = doc.DocumentNode;

        var userIdText = redirectedUrl.Split("=").Last();
        if (!int.TryParse(userIdText, out var userid))
        {
            if (TryGettingUserIdFromClickingHere(docNode, ref userid))
            {
                return null;
            }
        }

        var profile = new Profile
        {
            UserId = userid,
            UserName = userName,
            Gender = GetText(docNode, "Gender"),
            Age = GetText(docNode, "Age"),
            Weight = GetText(docNode, "Weight"),
            Height = GetText(docNode, "Height"),
            Location = GetText(docNode, "Location"),
            Orientation = GetText(docNode, "Orientation"),
            InterestedIn = GetText(docNode, "Hobbies"),
            Status = GetText(docNode, "Status"),
            Smoke = GetText(docNode, "Smoke"),
            Drink = GetText(docNode, "Drink"),
            Fetishes = GetText(docNode, "Fetishes"),
            Pets = GetText(docNode, "Pets"),
            JoinDate = GetText(docNode, "Member Since"),
            LastModified = GetText(docNode, "Updated")
        };

        UpdatePicture(userName, docNode, new Uri(redirectedUrl), profile);
        UpdateBio(docNode, profile);

        await UpdateProfileFromSearch(client, profile);

        return profile;
    }

    private static bool TryGettingUserIdFromClickingHere(HtmlNode docNode, ref int userid)
    {
        // Grab link that contains the user id
        var linkNode = docNode.SelectSingleNode("//a[contains(text(), \"clicking here\")]");
        var link = linkNode?.GetAttributeValue("href", "");
        if (string.IsNullOrEmpty(link))
        {
            return TryGettingUserIdFromReport(docNode, ref userid);
        }
        // https://www.literotica.com/stories/memberpage.php?uid=5470985&page=contact
        link = link.Replace("https://www.literotica.com/stories/memberpage.php?uid=", "").Replace("&page=contact", "");
        if (!int.TryParse(link, out userid))
        {
            return TryGettingUserIdFromReport(docNode, ref userid);
        }

        return false;
    }

    private static bool TryGettingUserIdFromReport(HtmlNode docNode, ref int userid)
    {
        // Grab link that contains the user id
        var linkNode = docNode.SelectSingleNode("//div/a/i/parent::a");
        var link = linkNode?.GetAttributeValue("href", "");
        if (string.IsNullOrEmpty(link))
        {
            return true;
        }

        // https://www.literotica.com/stories/report_profile.php?uid=5027871
        link = link.Replace("https://www.literotica.com/stories/report_profile.php?uid=", "");
        if (!int.TryParse(link, out userid))
        {
            return true;
        }

        return false;
    }

    private static async Task UpdateProfileFromSearch(HttpClient client, Profile profile)
    {
        var request = $"https://literotica.com/api/3/search/members?params=%7B%22q%22%3A%22{profile.UserName}%22%7D";
        var requestUri = new Uri(request);
        var responseString = await client.GetStringAsync(requestUri);

        var response = JsonSerializer.Deserialize<Response?>(responseString);

        var profileFromSearch = response?.Data.FirstOrDefault(p => p.UserName == profile.UserName);

        if (profileFromSearch != null)
        {
            // Copy some fields over
            profile.FavoriteStoriesCount = profileFromSearch.FavoriteStoriesCount;
            profile.FollowedStoriesCount = profileFromSearch.FollowedStoriesCount;
            profile.FollowersCount = profileFromSearch.FollowersCount;
            profile.FollowingsCount = profileFromSearch.FollowingsCount;
            profile.StoriesCount = profileFromSearch.StoriesCount;
            profile.PoemsCount = profileFromSearch.PoemsCount;
            profile.IllustrationsCount = profileFromSearch.IllustrationsCount;
            profile.SubmissionsCount = profileFromSearch.SubmissionsCount;
        }
    }

    private static void UpdateBio(HtmlNode docNode, Profile profile)
    {
        var bioNodes = docNode.SelectNodes("//div[contains(@class,'text__author')]/div/p");
        if (bioNodes is { Count: > 0 })
        {
            var sb = new StringBuilder();
            foreach (var bioNode in bioNodes)
            {
                if (bioNode.Name == "p")
                {
                    foreach (var node in bioNode.ChildNodes)
                    {
                        switch (node.NodeType)
                        {
                            case HtmlNodeType.Text:
                                sb.Append(HttpUtility.HtmlDecode(node.InnerText));
                                break;
                            case HtmlNodeType.Element when node.Name == "br":
                                sb.AppendLine();
                                break;
                            default:
                                throw new NotImplementedException($"Unhandled node {node.NodeType}/{node.Name}");
                        }
                    }
                }
                else
                {
                    throw new NotImplementedException($"Unhandled node {bioNode.Name}");
                }
            }

            profile.Bio = sb.ToString();
        }
    }

    private static void UpdatePicture(string userName, HtmlNode docNode, Uri requestUri, Profile profile)
    {
        var img = docNode.SelectSingleNode($"//img[@alt='userpic of {userName}']");
        if (img != null)
        {
            UpdatePicture(requestUri, profile, img, "_default");
        }
        else
        {
            img = docNode.SelectSingleNode("//img[@class='avPic']");
            if (img != null)
            {
                UpdatePicture(requestUri, profile, img);
            }
        }
    }

    private static void UpdatePicture(Uri requestUri, Profile profile, HtmlNode img, string useLarger = "_medium")
    {
        var src = img.GetAttributeValue("src", "");
        if (string.IsNullOrWhiteSpace(src))
        {
            return;
        }

        if (!string.IsNullOrWhiteSpace(useLarger))
        {
            src = src.Replace(useLarger, "_big");
        }

        var uri = new Uri(requestUri, src);
        profile.UserPic = uri.AbsoluteUri;
    }

    private static string GetText(HtmlNode docNode, string key)
    {
        var innerText = docNode.SelectSingleNode($"//p[contains(text(),'{key}')]/following-sibling::p")?.InnerText;
        if (innerText == null)
        {
            innerText = docNode.SelectSingleNode($"//p[text()[contains(.,'{key}')]]/following-sibling::span")
                ?.InnerText;
        }

        if (innerText == null)
        {
            innerText = docNode.SelectSingleNode($"//span[contains(text(),'{key}')]/preceding-sibling::span")
                ?.InnerText;
        }

        return innerText?.Replace("&nbsp;", "").Trim() ?? string.Empty;
    }

    public static async Task<string?> GetRedirectedUrl(string url)
    {
        //this allows you to set the settings so that we can get the redirect url
        var handler = new HttpClientHandler()
        {
            AllowAutoRedirect = false
        };

        using HttpClient client = new(handler);
        using var response = await client.GetAsync(url);
        using var content = response.Content;

        // ... Read the response to see if we have the redirected url
        if (response.StatusCode != System.Net.HttpStatusCode.Found && response.StatusCode != System.Net.HttpStatusCode.Moved && response.StatusCode != System.Net.HttpStatusCode.MovedPermanently)
        {
            return null;
        }

        var headers = response.Headers;
        if (headers.Location == null)
        {
            return null;
        }
        var location = headers.Location;
        if (location.IsAbsoluteUri)
        {
            return location.AbsoluteUri;
        }

        var baseUri = new Uri(url);
        var uri = new Uri(baseUri, location);
        return uri.AbsoluteUri;
    }
}